<?php

namespace Netzperfekt\SaasProptrend\Jobs;

use Filament\Notifications\Notification;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Netzperfekt\SaasBase\Models\User;
use Netzperfekt\SaasProptrend\Data\PropertyData;
use Netzperfekt\SaasProptrend\Enums\PropHistoryType;
use Netzperfekt\SaasProptrend\Enums\PropStatus;
use Netzperfekt\SaasProptrend\Models\ProptrendProjects;
use Netzperfekt\SaasProptrend\Models\ProptrendProp;
use Netzperfekt\SaasProptrend\Models\ProptrendPropsHistory;
use Netzperfekt\SaasProptrend\Models\ProptrendSource;

class ProcessPropertySave implements ShouldQueue
{
    use Queueable;

    public function __construct(private readonly PropertyData $propertyData, private readonly ?User $user)
    {
        //
    }

    public function handle(): void
    {
        $project = ProptrendProjects::with('sources')->find($this->propertyData->projectId);
        $source = ProptrendSource::find($this->propertyData->sourceId);

        if($source)
        {
            $fieldMappings = $source->field_mappings ?? [];

            foreach($this->propertyData->fields as $field)
            {
                if($field['path'])
                {
                    $fieldMappingPos = array_search($field['id'], array_column($fieldMappings, 'id'));

                    if($fieldMappingPos !== false)
                    {
                        $fieldMappings[$fieldMappingPos]['path'] = $field['path'];
                    }
                    else
                    {
                        $fieldMappings[] = ['id' => $field['id'], 'path' => $field['path']];
                    }
                }
            }

            $source->field_mappings = $fieldMappings;
            $source->save();
        }

        $prop = ProptrendProp::create([
            'team_id'    => $this->propertyData->tenantId,
            'project_id' => $project->id,
            'source_id'  => $source->id,
            'url'        => $this->propertyData->url,
            'place'      => $this->propertyData->place,
            'status'     => PropStatus::NeedsReview,
            'published_at' => now(),
        ]);

        $propHistory = ProptrendPropsHistory::create([
            'prop_id' => $prop->id,
            'type'    => PropHistoryType::Update,
            'html'    => $this->propertyData->html,
            'fields'  => $this->propertyData->fields,
        ]);

        if($this->user)
        {
            Notification::make()
                ->title(__('msg.prop_saved'))
                ->body($prop->place . ' / ' . $project->title)
                ->success()
                ->sendToDatabase($this->user);
        }

        // move to the very end - browsershot sometimes fails with exit(1) and this seems to be not catchable ;-(
        /*
        try {
            $screenshot = takeScreenshot($this->propertyData->url);

            if($screenshot)
            {
                $fileName = getMediaDirectory($prop->id, $this->propertyData->tenantId) . '/screen_' . Str::uuid()->toString() . '.png';
                Storage::disk(config('saas-proptrend.attachments_disk'))->put($fileName, $screenshot);

                $propHistory->media = [ $fileName ];
                $propHistory->save();
            }
        }
        catch(CouldNotTakeBrowsershot | ProcessFailedException | UnsuccessfulResponse | ElementNotFound $exception)
        {
            Log::error('browsershot failed', ['exception' => $exception->getMessage()]);
        }
        */
    }
}
