<?php

namespace Netzperfekt\SaasProptrend\Jobs;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Mail;
use Netzperfekt\SaasBase\Models\Team;
use Netzperfekt\SaasProptrend\Enums\FetchStatus;
use Netzperfekt\SaasProptrend\Enums\FieldType;
use Netzperfekt\SaasProptrend\Enums\PropHistoryType;
use Netzperfekt\SaasProptrend\Enums\PropStatus;
use Netzperfekt\SaasProptrend\Mail\PropertyUpdated;
use Netzperfekt\SaasProptrend\Models\ProptrendProp;
use Netzperfekt\SaasProptrend\Models\ProptrendPropsHistory;
use Symfony\Component\DomCrawler\Crawler;

class ProcessPropertyUpdate implements ShouldQueue
{
    use Queueable;

    public function __construct(private readonly ProptrendProp $property)
    {
        //
    }

    public function handle(): void
    {
        $allFields = $this->property?->latestHistoryEntry()?->fields ?? [];
        if(empty($allFields))
        {
            return;
        }
        $fieldMappings = $this->property->source->field_mappings;

        foreach($allFields as &$field)
        {
            $mappedFieldIndex = Collection
                ::make($fieldMappings)
                ->search(fn ($mappedField) => $mappedField['id'] == $field['id']);

            $field['path'] = '';
            if($mappedFieldIndex !== false)
            {
                $field['path'] = $fieldMappings[$mappedFieldIndex]['path'];
            }
        }

        $information = fetchUrl($this->property->url);
        $httpStatus = $information['_status'];

        if($httpStatus == FetchStatus::OK)
        {
            $html = $information['_html'];
            $fields = $this->extractFields($html, $allFields);

            $propHistory = ProptrendPropsHistory::create([
                'prop_id' => $this->property->id,
                'type'    => PropHistoryType::Update,
                'html'    => $html,
                'fields'  => $fields,
            ]);

            $this->property->setUpdatedAt(now());

            // move to the very end - browsershot sometimes fails with exit(1) and this seems to be not catchable ;-(
            /*
            try {
                $screenshot = takeScreenshot($this->property->url);

                if($screenshot)
                {
                    $fileName = getMediaDirectory($this->property->id, $this->property->team_id) . '/screen_' . Str::uuid()->toString() . '.png';
                    Storage::disk(config('saas-proptrend.attachments_disk'))->put($fileName, $screenshot);

                    $propHistory->media = [ $fileName ];
                    $propHistory->save();
                }
            }
            catch(CouldNotTakeBrowsershot | ProcessFailedException | UnsuccessfulResponse | ElementNotFound $exception)
            {
                Log::error('browsershot failed', ['exception' => $exception->getMessage()]);
            }
            */
        }

        else if($httpStatus == FetchStatus::NOT_FOUND)
        {
            ProptrendPropsHistory::create([
                'prop_id' => $this->property->id,
                'type'    => PropHistoryType::Removed,
                'html'    => '',
                'fields'  => [],
                'media'   => []
            ]);

            $this->property->status = PropStatus::Removed;
            $this->property->removed_at = now();
        }

        $this->property->save();

        if(count($this->property->getChangedFields()) > 0)
        {
            $this->notifyUserForUpdatedProperty($this->property);
        }
    }

    private function extractFields($html, $fields)
    {
        $crawler = new Crawler($html);

        $extractedData = [];
        foreach($fields as $field)
        {
            $fieldType = FieldType::from($field['type']);

            $value = $field['value'];
            if(! empty($field['path']))
            {
                $nodes = $crawler->filter($field['path']);
                if(count($nodes) > 0)
                {
                    $value = $nodes->innerText();
                    $value = sanitizeValue($value ?? '', $fieldType);
                }
            }

            $extractedData[] = [
                'id'    => $field['id'],
                'label' => $field['label'] ?? '',
                'type'  => $fieldType,
                'path'  => $field['path'] ?? '',
                'value' => $value
            ];
        }

        return $extractedData;
    }

    private function notifyUserForUpdatedProperty(ProptrendProp $property)
    {
        $allUsers = Team::find($property->team_id)->members;

        Mail::to($allUsers)->send(new PropertyUpdated($property));
    }
}
