<?php

namespace Netzperfekt\SaasProptrend\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Collection;
use Netzperfekt\SaasBase\Models\Team;
use Netzperfekt\SaasBase\Traits\HasDefaultScopes;
use Netzperfekt\SaasBase\Traits\HasGlobalModelObserver;
use Netzperfekt\SaasProptrend\Enums\FieldType;
use Netzperfekt\SaasProptrend\Enums\PropHistoryType;
use Netzperfekt\SaasProptrend\Enums\PropStatus;

class ProptrendProp extends Model
{
    use SoftDeletes;
    use HasGlobalModelObserver;
    use HasDefaultScopes;

    protected $table = 'proptrend_props';

    protected $fillable = [
        'id', 'deleted_at', 'created_at', 'updated_at', 'published_at', 'removed_at',
        'team_id',
        'project_id',
        'source_id',
        'status',

        'url',
        'tags',
        'comments_pro',
        'comments_contra',
        'rating',

        'longitude', 'latitude',
        'place', 'country', 'state',
        'location'
    ];

    protected $casts = [
        'status'       => PropStatus::class,
        'published_at' => 'datetime',
        'removed_at'   => 'datetime',
        'tags'         => 'array'
    ];

    protected $appends = [
        'location'
    ];

    public function team(): BelongsTo
    {
        return $this->belongsTo(Team::class);
    }

    public function project(): BelongsTo
    {
        return $this->belongsTo(ProptrendProjects::class);
    }

    public function source(): BelongsTo
    {
        return $this->belongsTo(ProptrendSource::class);
    }

    public function history(): HasMany
    {
        return $this->hasMany(ProptrendPropsHistory::class, 'prop_id', 'id');
    }

    public function getLocationAttribute(): array
    {
        return [
            "lat" => (float)$this->latitude,
            "lng" => (float)$this->longitude,
        ];
    }

    public function setLocationAttribute(?array $location): void
    {
        if (is_array($location))
        {
            $this->attributes['latitude'] = $location['lat'];
            $this->attributes['longitude'] = $location['lng'];
            unset($this->attributes['location']);
        }
    }

    public static function getLatLngAttributes(): array
    {
        return [
            'lat' => 'latitude',
            'lng' => 'longitude',
        ];
    }

    public static function getComputedLocation(): string
    {
        return 'location';
    }

    public function latestHistoryEntry()
    {
        return $this
            ->history()
            ->where('type', '=', PropHistoryType::Update)
            ->latest('created_at')
            ->first();
    }

    public function secondLatestHistoryEntry()
    {
        return $this
            ->history()
            ->where('type', '=', PropHistoryType::Update)
            ->orderBy('created_at', 'desc')
            ->skip(1)
            ->take(1)
            ->get()
            ->first();
    }

    public function getField(FieldType $fieldType): ?string
    {
        $latestHistoryEntry = $this->latestHistoryEntry();

        if ( ! $latestHistoryEntry)
        {
            return null;
        }

        $fieldIndex = Collection::make($latestHistoryEntry->fields)->search(function ($value) use ($latestHistoryEntry, $fieldType)
        {
            return $value['type'] == $fieldType->value;
        });

        if($fieldIndex !== false)
        {
            return $latestHistoryEntry?->fields[$fieldIndex]['value'] ?? null;
        }

        return null;
    }

    public function getChangedFields(bool $onlyChangedValues = true)
    {
        $latest = $this->latestHistoryEntry();
        $secondLatest = $this->secondLatestHistoryEntry();

        $changedFields = [];
        if($latest && $secondLatest)
        {
            $changedFields = ProptrendProp::getChangedFieldValues($latest, $secondLatest, $onlyChangedValues);
        }

        return $changedFields;
    }

    private function getChangedFieldValues(ProptrendPropsHistory $latest,
                                           ProptrendPropsHistory $secondLatest,
                                           bool $onlyChangedValues = true)
    {
        $changedFields = [];
        foreach($latest->fields as $field)
        {
            $oldFieldIndex = Collection
                ::make($secondLatest->fields)
                ->search(fn($oldField) => $oldField['id'] == $field['id']);

            $oldField = $secondLatest->fields[$oldFieldIndex];

            $fieldType = $field['type'];
            $oldValue = $oldField['value'] ?? null;
            $newValue = $field['value'] ?? null;

            if(($oldValue != $newValue) || $onlyChangedValues == false)
            {
                $changedFields[] = [
                    'id' => $field['id'],
                    'label' => $field['label'],
                    'type' => $fieldType,
                    'path' => $field['id'],
                    'old' => [
                        'id'        => $oldField['id'] ?? null,
                        'value'     => $oldValue . FieldType::from($fieldType)->getUnit(),
                        'date'      => $secondLatest->created_at,
                        'historyId' => $secondLatest->id
                    ],
                    'new' => [
                        'id'        => $field['id'] ?? null,
                        'value'     => $field['value'] . FieldType::from($fieldType)->getUnit(),
                        'date'      => $latest->created_at,
                        'historyId' => $latest->id
                    ],
                    'changed' => $oldValue != $newValue
                ];
            }
        }

        return $changedFields;
    }
}
