<?php

use Filament\Facades\Filament;
use Illuminate\Support\Facades\Http;
use Netzperfekt\SaasProptrend\Enums\FetchStatus;
use Netzperfekt\SaasProptrend\Enums\FieldType;
use Spatie\Browsershot\Browsershot;

const HTTP_CLIENT_USER_AGENT = 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/128.0.0.0 Safari/537.36';

function getMediaDirectory(string $propertyId, int $tenantId = 0): string
{
    return 'media/t' . $tenantId > 0 ? $tenantId : Filament::getTenant()->id . '/p' . $propertyId;
}

function calculateTotalPrice(?string $price, ?string $provision, string $state): float
{
    $price = intval($price);
    $provisionPercent = floatval(str_replace(',', '.', $provision));

    $stateTaxePercent = floatval(str_replace(',', '.', setting('state_taxes', [])->$state ?? 0));
    $notaryFeesPercent = floatval(str_replace(',', '.', setting('notary_fees', 0)));
    $landRegisterFeesPercent = floatval(str_replace(',', '.', setting('land_register_fees', 0)));

    $stateTax = $price / 100 * $stateTaxePercent;
    $notaryFees = $price / 100 * $notaryFeesPercent;
    $landRegisterFees = $price / 100 * $landRegisterFeesPercent;
    $provision = $price / 100 * $provisionPercent;

    return $price + $stateTax + $notaryFees + $landRegisterFees + $provision;
}

function fetchUrl(string $url): array
{
    try {
        $response = Http::withoutVerifying()->withUserAgent(HTTP_CLIENT_USER_AGENT)->withoutRedirecting()->get($url);

        $information = ['_html' => '', '_error' => '', '_status' => FetchStatus::UNKNOWN];

        if($response->ok())
        {
            $results = $response->body();
            $information = tryToGetInformationFrom($results);

            $information['_html'] = $results;
            $information['_status'] = FetchStatus::OK;
        }
        elseif($response->notFound() || $response->movedPermanently())
        {
            $information['_error'] = $response->status();
            $information['_status'] = FetchStatus::NOT_FOUND;
        }
    }
    catch (\Exception $exception)
    {
        $information['_error'] = $exception->getMessage();
        $information['_status'] = FetchStatus::ERROR;
    }

    return $information;
}

function tryToGetInformationFrom($results): array
{
    preg_match('/\\b\\d{5}\\s+[A-Za-zÄÖÜäöüß]+\\b/', $results, $matches);

    $information = [
        'place' => $matches[0] ?? ''
    ];

    return $information;
}

function sanitizeValue(string $value, FieldType $fieldType): string
{
    if($fieldType == FieldType::Number ||
        $fieldType == FieldType::Price ||
        $fieldType == FieldType::Provision ||
        $fieldType == FieldType::QmHouse)
    {
        $value = preg_replace('/\D/','', $value); // remove all non-numeric chars
    }

    return trim($value);
}

// spatie/browsershot
function takeScreenshot(string $url, int $width = 1024, int $height = 768, int $delay = 500, int $timeout = 60): ?string
{
    return null;

    // TODO erstmal abgeklemmt, klappt nicht richtig manchmal, der history entry wird dann nicht gespeichert!

    return Browsershot::url($url)
        ->windowSize($width, $height)
        ->setDelay($delay)
        ->timeout($timeout)
        ->fullPage()
        ->waitUntilNetworkIdle()
        ->dismissDialogs()
        ->noSandbox()
        ->ignoreHttpsErrors()
        ->setOption('headless', true)
        ->screenshot();
}
