<?php

namespace Netzperfekt\SaasTerve\Filament\Resources\ObjectResource;

use Cheesegrits\FilamentGoogleMaps\Fields\Map;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Split;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Support\Enums\FontWeight;
use Filament\Support\Enums\IconSize;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Str;
use Netzperfekt\SaasTerve\Enums\ObjectType;
use Netzperfekt\SaasTerve\Filament\Resources\ObjectResource\Pages\CreateObject;
use Netzperfekt\SaasTerve\Filament\Resources\ObjectResource\Pages\EditObject;
use Netzperfekt\SaasTerve\Filament\Resources\ObjectResource\Pages\ListObjects;
use Netzperfekt\SaasTerve\Models\TerveObject;

class ObjectResource extends Resource
{
    protected static ?string $model = TerveObject::class;

    protected static ?string $navigationIcon = 'fas-house';
    protected static ?string $recordTitleAttribute = 'title';

    public static function getNavigationGroup(): ?string
    {
        return __('terve-navigation-group');
    }

    public static function getNavigationBadge(): ?string
    {
        return static::getModel()::onlyMyRecords()->count();
    }

    public static function getModelLabel(): string
    {
        return __('terve-object');
    }

    public static function getPluralModelLabel(): string
    {
        return __('terve-objects');
    }

    public static function form(Form $form): Form
    {
        return $form
            ->columns(1)
            ->schema([
                Split::make([
                    Section::make()
                        ->compact()
                        ->schema([
                            Section::make('')
                                ->compact()
                                ->columns(3)
                                ->schema([
                                    TextInput::make('title')
                                        ->label(__('terve-title'))
                                        ->extraAttributes(['class' => 'font-semibold'])
                                        ->required()
                                        ->maxLength(255)
                                        ->autofocus()
                                        ->live(debounce: 1000)
                                        ->afterStateUpdated(function ($get, $set, ?string $state)
                                        {
                                            if(empty($get('slug'))) {
                                                $set('slug', Str::slug($state));
                                            }
                                        }),

                                    TextInput::make('slug')
                                        ->label(__('terve-slug'))
                                        ->required(),

                                    Grid::make()
                                        ->columns(4)
                                        ->schema([
                                        TextInput::make('street')
                                            ->label(__('terve-street'))
                                            ->maxLength(255),

                                        TextInput::make('zip')
                                            ->label(__('terve-zip'))
                                            ->maxLength(255),

                                        TextInput::make('city')
                                            ->label(__('terve-city'))
                                            ->maxLength(255),

                                        TextInput::make('country')
                                            ->label(__('terve-country'))
                                            ->maxLength(255)
                                    ]),

                                    Map::make('location')
                                        ->columns(1)
                                        ->label(__('terve-geolocation-map'))
                                        ->hint(__('terve-geolocation-map-hint'))
                                        ->mapControls([
                                            'mapTypeControl'    => true,
                                            'scaleControl'      => true,
                                            'streetViewControl' => false,
                                            'rotateControl'     => false,
                                            'fullscreenControl' => true,
                                            'searchBoxControl'  => false,
                                            'zoomControl'       => true,
                                        ])
                                        ->height(fn () => '400px')
                                        ->defaultZoom(17)
                                        ->autocomplete('contents.address')
                                        ->autocompleteReverse(true)
                                        ->reverseGeocode([
                                            'street' => '%n %S',
                                            'city' => '%L',
                                            'state' => '%A1',
                                            'zip' => '%z',
                                        ])
                                        ->reverseGeocodeUsing(function (callable $set, array $results) {
                                            $location = $results['results'][0]['geometry']['location'];
                                            $set('lat', $location['lat']);
                                            $set('lon', $location['lng']);
                                        })
                                        ->placeUpdatedUsing(function (callable $set, array $place) {
                                            $set('lat', $place['geometry']['location']['lat']);
                                            $set('lon', $place['geometry']['location']['lng']);
                                        })
                                        ->defaultLocation([52.5200066, 13.4049540])
                                        ->draggable()
                                        ->clickable(true),

                                    TextInput::make('email')
                                        ->label(__('terve-email'))
                                        ->email()
                                        ->required()
                                        ->maxLength(255),

                                    TextInput::make('phone')
                                        ->label(__('terve-phone'))
                                        ->maxLength(255),

                                    TextInput::make('homepage')
                                        ->url()
                                        ->label(__('terve-homepage'))
                                        ->maxLength(255)
                                ])
                        ]),

                    Section::make()
                        ->compact()
                        ->schema([
                            Select::make('type')
                                ->label(__('terve-type'))
                                ->required()
                                ->options(ObjectType::class)
                                ->live()
                        ])->grow(false)
                ])
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->defaultPaginationPageOption(25)
            ->groups([
                Tables\Grouping\Group::make('type')
                    ->label(__('terve-type'))
                    ->collapsible()
            ])

            ->columns([
                Tables\Columns\TextColumn::make('title')
                    ->label(__('terve-title'))
                    ->weight(FontWeight::SemiBold)
                    ->searchable(),

                Tables\Columns\TextColumn::make('city')
                    ->label(__('terve-city'))
                    ->searchable(),

                Tables\Columns\TextColumn::make('country')
                    ->label(__('terve-country'))
                    ->searchable(),

                Tables\Columns\TextColumn::make('type')
                    ->label(__('terve-type'))
                    ->badge()
                    ->color('warning')
            ])

            ->filters([
                Tables\Filters\TrashedFilter::make(),

                Tables\Filters\SelectFilter::make('type')
                    ->label(__('terve-type'))
                    ->options(ObjectType::class)
            ])
            
            ->actions([
                Tables\Actions\EditAction::make()->label('')->tooltip('Bearbeiten')->iconSize(IconSize::Medium),
                Tables\Actions\DeleteAction::make()->label('')->tooltip('Löschen')->iconSize(IconSize::Medium),
                Tables\Actions\ForceDeleteAction::make()->label('')->tooltip('Endgültig löschen')->iconSize(IconSize::Medium),
                Tables\Actions\RestoreAction::make()->label('')->tooltip('Wiederherstellen')->iconSize(IconSize::Medium)
            ])
            
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                ]),
            ])
            
            ->emptyStateActions([
                Tables\Actions\CreateAction::make(),
            ]);
    }

    public static function getRelations(): array
    {
        return [
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => ListObjects::route('/'),
            'create' => CreateObject::route('/create'),
            'edit' => EditObject::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): \Illuminate\Database\Eloquent\Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }
}
